	function h = FLAT_BP_FIR_2(N,L,woT,dsl, dsu)
	
	% A program for the design of symmetric bandpass FIR filters with 
	% flat monotonically decreasing passbands (on either side of woT)
	% and equiripple stopbands.
	%
	
	% input
	%  N  : filter order
	%  L  : degree of flatness
	%  woT : passband frequency of flatness
	%  dsu : upper bound in the stopbands
	%  dsl : lower boudn in the stopbands
	%
	% Author: Ivan W. Selesnick, Rice University
	%
	% Modified: LW 2004-07-18
	
	% Reference:
	% 	I.W. Selesnick and C.S. Burrus, Exchange Algorithms for the Design 
	% 	of Linear Phase FIR Filters and Differentiators Having Flat
	%	Monotonic Passbands and Equiripple Stopbands
	%	IEEE Trans. on Cicuits and Systems, Part II, Vol. 43, No. 9, 
	%	pp. 671-675, Sept 1996.
	
	N = N+1; % filter length
	if dsl >= 0
		disp('dsl must be be negative')
		return
	end
	if (rem(N,2) == 0) | (rem(L,4) ~= 0)
		disp('N must be even and L must be divisible by 4')
		return
	else
		g = 2^ceil(log2(4*N));		% number of grid points
		SN = 1e-9;                  % SN : SMALL NUMBER
		q  = (N-L+1)/2;
		w  = [0:g]'*pi/g;
		ws1 = woT*0.8;
		a = 5; b = 1;
		ws2 = (a*woT+b*pi)/(a+b);	
		d = ws1/(pi-ws2);    	% q1 : number of ref. set freq. in 1st stopband
		q1 = round(q/(1+1/d));  % q2 : number of ref. set freq. in 2nd stopband
		if q1 == 0
			q1 = 1;
		elseif q1 == q
			q1 = q - 1;
		end
		q2 = q - q1;	
		if q1 == 1;
	        rs1 = ws1;
		else
	        rs1 = [0:q1-1]'*(ws1/(q1-1));
		end
		if q2 == 1
	        rs2 = ws2;
		else
	        rs2 = [0:q2-1]'*(pi-ws2)/(q2-1)+ws2;
		end
		rs = [rs1; rs2];		
		Y1 = [dsl*(1-(-1).^(q1:-1:1))/2 + dsu*((-1).^(q1:-1:1)+1)/2]';
		Y2 = [dsl*(1-(-1).^(1:q2))/2 + dsu*((-1).^(1:q2)+1)/2]';
		Y = [Y1; Y2];	
		n  = 0:q-1;
		Z   = zeros(2*(g-q)+1,1);
		A1  = (-1)^(L/2) * ((cos(woT)-cos(w))/2).^(L/2);
		A1r = (-1)^(L/2) * ((cos(woT)-cos(rs))/2).^(L/2);
		it = 0;
		while 1 & (it < 20)
			a2 = cos(rs*n)\[(Y-1)./A1r];
			A2 = real(fft([a2(1);
			a2(2:q)/2;Z;
			a2(q:-1:2)/2])); 
			A2 = A2(1:g+1);
			A = 1 + A1.*A2;
			ri = sort([LOCAL_MAX(A); LOCAL_MAX(-A)]);
			lri = length(ri);
	        if lri ~= q+1
				if abs(A(ri(lri))-A(ri(lri-1))) < abs(A(ri(1))-A(ri(2)))
					ri(lri) = [];
				else
					ri(1) = [];
				end
			end
			rs = (ri-1)*pi/g;
			[temp, k] = min(abs(rs - woT)); rs(k) = [];
			q1 = sum(rs < woT);
			q2 = sum(rs > woT);	
			Y1 = [dsl*(1-(-1).^(q1:-1:1))/2 + dsu*((-1).^(q1:-1:1)+1)/2]';
			Y2 = [dsl*(1-(-1).^(1:q2))/2 + dsu*((-1).^(1:q2)+1)/2]';
			Y = [Y1; Y2];
			A1r = (-1)^(L/2) * ((cos(woT)-cos(rs))/2).^(L/2);
			Ar = 1 + (cos(rs*n)*a2).* A1r;
			Err = max([max(Ar)-dsu,dsl-min(Ar)]);
			fprintf(1,'    Err = %18.15f\n',Err);
			if Err < SN, fprintf(1,'\n    Converged \n'), break, end
			it = it + 1;
		end
		h2 = [a2(q:-1:2)/2; a2(1); a2(2:q)/2];
		h = h2;
		for k = 1:L/2
			h = conv(h,[1 -2*cos(woT) 1]')/4;
		end
		h((N+1)/2) = h((N+1)/2) + 1;	
	end
	
